'use strict';

const DEFAULT_SETTINGS = Object.freeze({
  apiBaseUrl: 'https://pass.gzutil.com',
  email: '',
  token: '',
  autoFill: true,
  autoCapture: true,
  suggestionsLimit: 10
});

// TTL bajo para detectar rápidamente cambios de despliegue sin saturar la API.
const CACHE_TTL_MS = 60 * 1000;
const domainCache = new Map();
let lastSyncInfo = { timestamp: null, domain: null };
let lastError = null;

async function getSettings() {
  const stored = await chrome.storage.local.get('gpwSettings');
  return {
    ...DEFAULT_SETTINGS,
    ...(stored.gpwSettings || {})
  };
}

async function saveSettings(partial) {
  const next = {
    ...(await getSettings()),
    ...partial
  };
  await chrome.storage.local.set({ gpwSettings: next });
  return next;
}

function clearCache(domain = null) {
  if (domain) {
    domainCache.delete(domain);
    return;
  }
  domainCache.clear();
}

async function ensureDefaults() {
  const stored = await chrome.storage.local.get('gpwSettings');
  if (!stored.gpwSettings) {
    await chrome.storage.local.set({ gpwSettings: { ...DEFAULT_SETTINGS } });
  }
}

async function apiRequest({ method = 'GET', body = undefined, query = {}, settings = null }) {
  const cfg = settings ?? (await getSettings());
  if (!cfg.token) {
    throw new Error('Inicia sesión en las opciones para obtener tu token de integración.');
  }

  const base = (cfg.apiBaseUrl || DEFAULT_SETTINGS.apiBaseUrl).replace(/\/+$/, '');
  const url = new URL(base + '/api/extensions/vault.php');
  Object.entries(query).forEach(([key, value]) => {
    if (value === undefined || value === null || value === '') {
      return;
    }
    url.searchParams.set(key, String(value));
  });

  const headers = new Headers({
    'Content-Type': 'application/json',
    'X-Integration-Token': cfg.token
  });

  const response = await fetch(url.toString(), {
    method,
    headers,
    body: body ? JSON.stringify(body) : undefined
  });

  if (!response.ok) {
    let message = `Error ${response.status}`;
    try {
      const details = await response.json();
      if (typeof details?.error === 'string') {
        message = details.error;
      } else if (typeof details?.message === 'string') {
        message = details.message;
      }
    } catch (_) {
      message = await response.text();
    }
    throw new Error(message || 'No se pudo contactar con la API');
  }

  const text = await response.text();
  if (text === '') {
    return null;
  }

  return JSON.parse(text);
}

async function loginWithCredentials({ email, password }) {
  const settings = await getSettings();
  const base = (settings.apiBaseUrl || DEFAULT_SETTINGS.apiBaseUrl).replace(/\/+$/, '');
  const url = new URL(base + '/api/extensions/login.php');

  const response = await fetch(url.toString(), {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({ email, password })
  });

  if (!response.ok) {
    let message = `Error ${response.status}`;
    try {
      const details = await response.json();
      if (typeof details?.error === 'string') {
        message = details.error;
      } else if (typeof details?.message === 'string') {
        message = details.message;
      }
    } catch (_) {
      message = await response.text();
    }
    throw new Error(message || 'No se pudo iniciar sesión');
  }

  const data = await response.json();
  const token = data?.token || '';
  if (!token) {
    throw new Error('La respuesta de login no incluyó un token de integración');
  }

  await saveSettings({
    email: email || settings.email || '',
    token
  });

  clearCache();

  return {
    user: data.user || null
  };
}

async function fetchSuggestions(domain, { force = false } = {}) {
  const normalizedDomain = (domain || '').toLowerCase();
  if (!normalizedDomain) {
    throw new Error('Dominio no válido');
  }

  if (!force) {
    const cached = domainCache.get(normalizedDomain);
    if (cached && Date.now() - cached.timestamp < CACHE_TTL_MS) {
      return { items: cached.items, cached: true };
    }
  }

  const settings = await getSettings();
  const limit = Math.max(1, Math.min(50, Number(settings.suggestionsLimit) || DEFAULT_SETTINGS.suggestionsLimit));
  const data = await apiRequest({
    query: { domain: normalizedDomain, limit: String(limit) },
    settings
  });

  const items = Array.isArray(data?.items) ? data.items : [];
  domainCache.set(normalizedDomain, { items, timestamp: Date.now() });
  lastSyncInfo = {
    timestamp: data?.generated_at || new Date().toISOString(),
    domain: normalizedDomain
  };
  lastError = null;
  return { items, cached: false };
}

async function saveCredential(payload) {
  const settings = await getSettings();
  const site =
    (payload?.site && String(payload.site).trim()) ||
    (payload?.domain ? `https://${payload.domain}` : settings.apiBaseUrl);
  await apiRequest({
    method: 'POST',
    body: {
      site,
      username: payload?.username || '',
      password: payload?.password || '',
      notes: payload?.notes || null
    },
    settings
  });
  if (payload?.domain) {
    clearCache(payload.domain.toLowerCase());
  }
}

async function updateCredential(payload) {
  const settings = await getSettings();
  const id = payload?.id;
  if (!id) {
    throw new Error('Se requiere el id de la credencial para actualizarla');
  }

  const body = { id };
  if (payload?.password) {
    body.password = payload.password;
  }
  if (payload?.username !== undefined) {
    body.username = payload.username;
  }
  if (payload?.site !== undefined) {
    body.site = payload.site;
  }
  if (payload?.notes !== undefined) {
    body.notes = payload.notes;
  }

  await apiRequest({
    method: 'PUT',
    body,
    settings
  });

  if (payload?.domain) {
    clearCache(payload.domain.toLowerCase());
  }
}

async function incrementUsage(payload) {
  const settings = await getSettings();
  const id = payload?.id;
  if (!id) {
    throw new Error('Se requiere el id de la credencial para incrementar uso');
  }

  await apiRequest({
    method: 'PATCH',
    body: {
      action: 'increment_usage',
      id
    },
    settings
  });

  // Limpiar caché para que la próxima vez se obtenga el orden actualizado
  if (payload?.domain) {
    clearCache(payload.domain.toLowerCase());
  }
}

async function handleMessage(message) {
  switch (message?.type) {
    case 'gpw:getSuggestions':
      return fetchSuggestions(message.domain, { force: Boolean(message.force) });
    case 'gpw:saveCredential':
      await saveCredential(message.payload);
      return { status: 'saved' };
    case 'gpw:updateCredential':
      await updateCredential(message.payload);
      return { status: 'updated' };
    case 'gpw:incrementUsage':
      await incrementUsage(message.payload);
      return { status: 'usage_incremented' };
    case 'gpw:getStatus':
      return {
        hasToken: Boolean((await getSettings()).token),
        lastSync: lastSyncInfo,
        lastError
      };
    case 'gpw:refreshDomain':
      return fetchSuggestions(message.domain, { force: true });
    case 'gpw:getSettings':
      return getSettings();
    case 'gpw:setSettings':
      return saveSettings(message.payload || {});
    case 'gpw:loginWithCredentials':
      return loginWithCredentials({
        email: message.email || '',
        password: message.password || ''
      });
    case 'gpw:testConnection': {
      await apiRequest({ query: { limit: '1' } });
      return { ok: true };
    }
    default:
      throw new Error('Acción no soportada');
  }
}

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message)
    .then((result) => {
      sendResponse({ ok: true, data: result });
    })
    .catch((error) => {
      lastError = error.message;
      sendResponse({ ok: false, error: error.message });
    });
  return true;
});

chrome.storage.onChanged.addListener((changes, areaName) => {
  if (areaName === 'local' && changes.gpwSettings) {
    clearCache();
  }
});

chrome.runtime.onInstalled.addListener((details) => {
  ensureDefaults().catch((error) => {
    console.error('[GestorPass] No se pudieron establecer los valores por defecto', error);
  });

  // Al instalar por primera vez, abrir automáticamente la página de opciones
  if (details && details.reason === 'install') {
    chrome.runtime.openOptionsPage();
  }
});

